function result = LWM_RND(oprice,strike, rf, TTM,Ft,Mmax, weights,LWM_options)
%==========================================================================================
%The function estimate the parameters of the LWM-based RND based on a
%cross-section of call option prices using the global search algorithm 
%based on Li, Nolte and Pham (2023).
%
%INPUT:
%   oprice: N-by-1 call option prices
%   strike: N-by-1 strike prices of the options
%   rf: risk-free rate
%   TTM: time to maturity of the options (in years)
%   Ft: underlying futures price
%   Mmax: maximum number of mixtures for the LWM method. Default = 5
%   weights: N-by-1 weight vector. If not provided, assumed equal weights
%   LWM_options: options to fine-tune the LWM-based RND estimation. If not
%   provided, default options are used. See the function LWM_options_setup.m for
%   details.
%
%OUTPUT:
%   result: a struct consisting of
%          fvals: a Mmax-by-5 matrix, with each row corresponding to the
%                 number of mixtures (M) from 1 to Mmax, and 
%                 columns containing in order: 
%                     1) number of mixtures M, 
%                     2) objective function value with M1+1 Lognormal and M2 Weibull
%                     3) objective function value with M1 Lognormal and M2+1 Weibull
%                     4) number of lognormal densities in the mixture
%                     5) number of Weilbull densitites in the mixture
%         M_x_estpar: estimated parameters of the M-LWM model, with x = 2:5
%==========================================================================================
% This ver: 2023/05/24
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2023). Parametric Risk-Neutral 
%          Density Estimation via Finite Lognormal-Weibull Mixtures
%========================================================================================== 

if nargin < 6
    Mmax = 5;
end

if nargin < 7
    weights = ones(size(oprice));
end

if nargin < 8
    LWM_options = LWM_options_setup();
end

% Initialization 
LWM_ini = LWM_RND_initialization(oprice,strike, rf, TTM,Ft,weights);
M1=0;M2=0;
% Struct to store results
result = struct;
fvalk = LWM_ini.fvalk*LWM_options.objval_scale;
fvals = LWM_ini.fvals*LWM_options.objval_scale;
result.fvals = zeros(Mmax,5);
result.fvals(1,:)=[1 fvals  fvalk M1+ double(fvalk>=fvals) M2+double(fvalk<fvals)];
% Branching
for M=2:Mmax
    LWM_options.fval_ub = min(fvalk,fvals);
    M1 = M1 + double(fvalk>=fvals);
    M2 = M2 + double(fvalk<fvals);
    [estpars,fvals]=LWM_RND_core(M1+1,M2,oprice,strike, rf, TTM,Ft, weights,LWM_ini,LWM_options);
    [estpark,fvalk]=LWM_RND_core(M1,M2+1,oprice,strike, rf, TTM,Ft, weights,LWM_ini,LWM_options);
    skchoice = [fvalk>=fvals; fvalk<fvals];
    estpar = [estpars estpark]*skchoice;
    result.('M_'+string(M)+'_estpar') = estpar;
    result.fvals(M,:)=  [M fvals  fvalk M1+ double(fvalk>=fvals) M2+double(fvalk<fvals)];
    Mvec=[M1 M2]+skchoice';
    LWM_ini.x0s = [estpar(1:Mvec(1)) estpar(M+1:M+Mvec(1)) estpar(2*M+1:2*M+Mvec(1))];
    LWM_ini.x0k = [estpar(Mvec(1)+1:M) estpar(M+Mvec(1)+1:2*M) estpar(2*M+Mvec(1)+1:3*M)];
end

end

